﻿/* -LICENSE-START-
** Copyright (c) 2017 Blackmagic Design
**
** Permission is hereby granted, free of charge, to any person or organization
** obtaining a copy of the software and accompanying documentation covered by
** this license (the "Software") to use, reproduce, display, distribute,
** execute, and transmit the Software, and to prepare derivative works of the
** Software, and to permit third-parties to whom the Software is furnished to
** do so, all subject to the following:
** 
** The copyright notices in the Software and this entire statement, including
** the above license grant, this restriction and the following disclaimer,
** must be included in all copies of the Software, in whole or in part, and
** all derivative works of the Software, unless such copies or derivative
** works are solely in the form of machine-executable object code generated by
** a source language processor.
** 
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
** SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
** FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
** -LICENSE-END-
*/
using System;
using DeckLinkAPI;
using System.Collections.Generic;

namespace CapturePreviewCSharp
{
    public delegate void DeckLinkInputSignalHandler(bool inputSignal);
    public delegate void DeckLinkFormatChangedHandler(IDeckLinkDisplayMode newDisplayMode);
    class DeckLinkDevice : IDeckLinkInputCallback, IEnumerable<IDeckLinkDisplayMode>
    {
        private IDeckLink           m_deckLink;
        private IDeckLinkInput      m_deckLinkInput;
        private bool                m_applyDetectedInputMode = true;
        private bool                m_currentlyCapturing = false;
        private bool                m_validInputSignal = false;

        public DeckLinkDevice(IDeckLink deckLink)
        {
            m_deckLink = deckLink;

            // Get input interface
            m_deckLinkInput = (IDeckLinkInput)m_deckLink;
        }

        public event DeckLinkInputSignalHandler InputSignalChanged;
        public event DeckLinkFormatChangedHandler InputFormatChanged;

        public IDeckLink deckLink
        {
            get { return m_deckLink; }
        }

        public string deviceName
        {
            get
            {
                string deviceName;
                m_deckLink.GetDisplayName(out deviceName);
                return deviceName;
            }
        }

        public bool supportsFormatDetection
        {
            get
            {
                int flag;
                var deckLinkAttributes = (IDeckLinkAttributes)m_deckLink;
                deckLinkAttributes.GetFlag(_BMDDeckLinkAttributeID.BMDDeckLinkSupportsInputFormatDetection, out flag);
                return flag != 0;
            }
        }

        public bool isCapturing
        {
            get { return m_currentlyCapturing; }
        }

        void IDeckLinkInputCallback.VideoInputFormatChanged(_BMDVideoInputFormatChangedEvents notificationEvents, IDeckLinkDisplayMode newDisplayMode, _BMDDetectedVideoInputFormatFlags detectedSignalFlags)
        {
            // Restart capture with the new video mode if told to
            if (! m_applyDetectedInputMode)
                return;

            var pixelFormat = _BMDPixelFormat.bmdFormat10BitYUV;
            if (detectedSignalFlags.HasFlag(_BMDDetectedVideoInputFormatFlags.bmdDetectedVideoInputRGB444))
                pixelFormat = _BMDPixelFormat.bmdFormat10BitRGB;

            // Stop the capture
            m_deckLinkInput.StopStreams();

            // Set the video input mode
            m_deckLinkInput.EnableVideoInput(newDisplayMode.GetDisplayMode(), pixelFormat, _BMDVideoInputFlags.bmdVideoInputEnableFormatDetection);

            // Start the capture
            m_deckLinkInput.StartStreams();

            InputFormatChanged(newDisplayMode);
        }

        void IDeckLinkInputCallback.VideoInputFrameArrived(IDeckLinkVideoInputFrame videoFrame, IDeckLinkAudioInputPacket audioPacket)
        {
            if (videoFrame != null)
            {
                bool inputSignal = videoFrame.GetFlags().HasFlag(_BMDFrameFlags.bmdFrameHasNoInputSource);
                if (inputSignal != m_validInputSignal)
                {
                    m_validInputSignal = inputSignal;
                    InputSignalChanged(m_validInputSignal);
                }
            }

            System.Runtime.InteropServices.Marshal.ReleaseComObject(videoFrame);
        }

        IEnumerator<IDeckLinkDisplayMode> IEnumerable<IDeckLinkDisplayMode>.GetEnumerator()
        {
            IDeckLinkDisplayModeIterator displayModeIterator;
            m_deckLinkInput.GetDisplayModeIterator(out displayModeIterator);
            return new DisplayModeEnum(displayModeIterator);
        }

        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            throw new InvalidOperationException();
        }

        public void StartCapture(IDeckLinkDisplayMode displayMode, IDeckLinkScreenPreviewCallback screenPreviewCallback, bool applyDetectedInputMode)
        {
            if (m_currentlyCapturing)
                return;

            var videoInputFlags = _BMDVideoInputFlags.bmdVideoInputFlagDefault;

            m_applyDetectedInputMode = applyDetectedInputMode;
            m_validInputSignal = false;

            // Enable input video mode detection if the device supports it
            if (supportsFormatDetection && m_applyDetectedInputMode)
                videoInputFlags |= _BMDVideoInputFlags.bmdVideoInputEnableFormatDetection;

            // Set the screen preview
            m_deckLinkInput.SetScreenPreviewCallback(screenPreviewCallback);

            // Set capture callback
            m_deckLinkInput.SetCallback(this);

            // Set the video input mode
            m_deckLinkInput.EnableVideoInput(displayMode.GetDisplayMode(), _BMDPixelFormat.bmdFormat8BitYUV, videoInputFlags);

            // Start the capture
            m_deckLinkInput.StartStreams();

            m_currentlyCapturing = true;
        }

        public void StopCapture()
        {
            if (!m_currentlyCapturing)
                return;

            RemoveAllListeners();

            // Stop the capture
            m_deckLinkInput.StopStreams();

            // disable callbacks
            m_deckLinkInput.SetScreenPreviewCallback(null);
            m_deckLinkInput.SetCallback(null);

            m_currentlyCapturing = false;
        }

        void RemoveAllListeners()
        {
            InputSignalChanged = null;
            InputFormatChanged = null;
        }
    }

    class DisplayModeEnum : IEnumerator<IDeckLinkDisplayMode>
    {
        private IDeckLinkDisplayModeIterator m_displayModeIterator;
        private IDeckLinkDisplayMode m_displayMode;

        public DisplayModeEnum(IDeckLinkDisplayModeIterator displayModeIterator)
        {
            m_displayModeIterator = displayModeIterator;
        }

        IDeckLinkDisplayMode IEnumerator<IDeckLinkDisplayMode>.Current
        {
            get { return m_displayMode; }
        }

        bool System.Collections.IEnumerator.MoveNext()
        {
            m_displayModeIterator.Next(out m_displayMode);
            return m_displayMode != null;
        }

        void IDisposable.Dispose()
        {
        }

        object System.Collections.IEnumerator.Current
        {
            get { return m_displayMode; }
        }

        void System.Collections.IEnumerator.Reset()
        {
            throw new InvalidOperationException();
        }
    }

    public delegate void DeckLinkDiscoveryHandler(IDeckLink decklinkDevice);
    class DeckLinkDeviceDiscovery : IDeckLinkDeviceNotificationCallback
    {
        private IDeckLinkDiscovery      m_deckLinkDiscovery;

        public event DeckLinkDiscoveryHandler DeviceArrived;
        public event DeckLinkDiscoveryHandler DeviceRemoved;

        public DeckLinkDeviceDiscovery()
        {
            m_deckLinkDiscovery = new CDeckLinkDiscovery();
        }
        ~DeckLinkDeviceDiscovery()
        {
            Disable();
        }

        public void Enable()
        {
            m_deckLinkDiscovery.InstallDeviceNotifications(this);
        }
        public void Disable()
        {
            m_deckLinkDiscovery.UninstallDeviceNotifications();
        }

        void IDeckLinkDeviceNotificationCallback.DeckLinkDeviceArrived(IDeckLink deckLinkDevice)
        {
            DeviceArrived(deckLinkDevice);
        }

        void IDeckLinkDeviceNotificationCallback.DeckLinkDeviceRemoved(IDeckLink deckLinkDevice)
        {
            DeviceRemoved(deckLinkDevice);
        }
    }
}
